/******************************************************************************
 *
 * tw_extension.c - Extension for the Tivaware Driver Library for TI Tiva C
 *  microcontrollers. For use with the TI Tiva C Launchpad EK-TM4C123GXL
 *  development board.
 *
 *  Author: Curtis Mayberry
 *  Georgia Tech IMEMS
 *  rev1 March 2014
 *
 *  Originally written for the MRIG gyroscope project
 *
 *  This work is licensed under the Creative Commons Attribution-ShareAlike 3.0
 *  Unported License. To view a copy of this license, visit
 *  http://creativecommons.org/licenses/by-sa/3.0/ or send a letter to Creative
 *  Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.
 *
 ******************************************************************************/

#include <stdbool.h>
#include <stdint.h>
#include "inc/hw_memmap.h"
#include "inc/hw_ssi.h"
#include "inc/hw_types.h"
#include "inc/hw_ints.h"
#include "inc/hw_timer.h"

// Tivaware
#include "driverlib/rom.h"
#include "driverlib/sysctl.h"
#include "driverlib/debug.h"
#include "driverlib/flash.h"
#include "driverlib/fpu.h"
#include "driverlib/uart.h"
#include "driverlib/gpio.h"
#include "driverlib/pin_map.h"
#include "driverlib/interrupt.h"
#include "driverlib/timer.h"
#include "driverlib/udma.h"
#include "driverlib/ssi.h"

// Launchpad Drivers
#ifdef PART_TM4C123GH6PM // EK-TM4C123GXL
#include "examples/boards/ek-tm4c123gxl/drivers/rgb.h"
#endif

// GTBE Lib
#include "tw_extension.h"

//*****************************************************************************
//
//! \internal
//! Checks an SSI base address.
//!
//! \param ui32Base specifies the SSI module base address.
//!
//! This function determines if a SSI module base address is valid.
//!
//! \return Returns \b true if the base address is valid and \b false
//! otherwise.
//
//*****************************************************************************
#ifdef DEBUG
static bool
_SSIBaseValid(uint32_t ui32Base)
{
    return((ui32Base == SSI0_BASE) || (ui32Base == SSI1_BASE) ||
           (ui32Base == SSI2_BASE) || (ui32Base == SSI3_BASE));
}
#endif

/*******************
 * Flash Functions *
 *******************/

/**
 * Initializes the Flash
 *  Protects the part of the flash memory from address codeStart through
 *   codeStart + codeReserveLength by setting to only execute (FlashExecuteOnly)
 *  Sets up the flash interrupt in case the program attempts to access the
 *   protected flash portions.  Make sure the ISR is added to the NVIC table
 *
 *  \ Make sure the NVIC is not contained in the protected code memory
 *
 *  \param codeStart - start location of flash memory that is to be reserved
 *  	it must land on a 2KB boundry
 *  \param codeReserveLength - length of code to protect in flash memory
 *  	it must land on a 2KB boundry, set to 0x0 to skip protection
 *  \param dataStart - start location of flash memory that is to be erased
 *  	it must land on a 1KB boundry
 *  \param dataEraseLength  - length of data to erase in flash memory
 *  	it must land on a 1KB boundry, set to 0x0 to skip erasure
 **/
int32_t twe_initFlash(uint32_t codeStart, uint32_t codeReserveLength,
					  uint32_t dataStart, uint32_t dataEraseLength) {
	int32_t status = 0;
	//protect Code area
	status = twe_protectFlashRange(codeStart, codeReserveLength, FlashExecuteOnly);
	// Erase Data in flash memory
	status = twe_eraseFlashRange(dataStart, dataEraseLength);
	//setup Flash Interrupt to handle any improper write attempts to flash
	FlashIntEnable(FLASH_INT_PROGRAM);
	return status;
}

/**
 * Erases the indicated flash memory
 *  \param dataStart - start location of flash memory that is to be erased
 *  	it must land on a 1KB boundry
 *  \param dataEraseLength  - length of data to erase in flash memory
 *  	it must land on a 1KB boundry, set to 0x0 to skip erasure
 **/
int32_t twe_eraseFlashRange(uint32_t startAddress, uint32_t eraseLength) {
	int32_t status = 0;
	uint32_t modCheck;
	uint32_t block;
	modCheck = startAddress - startAddress/0x400;
	modCheck = modCheck + eraseLength/0x400;
	ASSERT(modCheck == 0);
	if(eraseLength > 0x0) {
		for(block = 0; block < (eraseLength/0x400); block++) {
			status = FlashErase(startAddress + 0x400 * block) // Erases each 1KB block
			ASSERT(status == 0);
		}
	}
	return status;
}

/**
 * Protects the indicated flash memory
 *  \param codeStart - start location of flash memory that is to be reserved
 *  	it must land on a 2KB boundry
 *  \param protectLength - length of code to protect in flash memory
 *  	it must land on a 2KB boundry, set to 0x0 to skip protection
 **/
int32_t twe_protectFlashRange(uint32_t startAddress, uint32_t protectLength, tFlashProtection eProtect) {
	int32_t status = 0;
	uint32_t modCheck;
	uint32_t block;
	// Check codeStart and codeReserveLength to ensure the protected memory
	//  block starts and ends on a 2KB boundry
	modCheck = startAddress - startAddress/0x800;
	modCheck = modCheck + protectLength/0x800;
	ASSERT(modCheck == 0);
	// Setup code protection
	if(protectLength > 0x0) {
		for(block = 0; block < (protectLength/0x800); block++) {
			status = FlashProtectSet(0x800 * block, eProtect);
			ASSERT(status == 0);
		}
	}
	return status;
}

/**
 * Interrupt handler (ISR) that executes when an improper write to flash is attempted
 **/
void twe_FLASH_badWriteISR(void) {
	while(1) {
		// Do nothing - save state for examination
	}
}

/******************
 * FPU Functions *
 ******************/

/**
 * Initializes the FPU
 **/
void twe_initFPU(void) {
	FPUEnable();
}

/**
 * Initializes the FPU with lazy stacking enabled
 **/
void twe_initFPUlazy(void) {
	// Lazy Stacking increases interrupt latency and stack usage
	//  (only need if doing floating pt. in interrupts)
	FPULazyStackingEnable();
	FPUEnable();
}

/******************
 * UART Functions *
 ******************/

/**
 * Initializes UART0
 *  UART0 is connected to the stellaris virtual serial port through the USB connection
 *  Configuration:
 *   8 data bits
 *   one stop bit
 *   no parity
 *   115200 BAUD
 **/
void twe_initUART0(void) {
	SysCtlPeripheralEnable(SYSCTL_PERIPH_UART0);
	SysCtlPeripheralEnable(SYSCTL_PERIPH_GPIOA);

	GPIOPinConfigure(GPIO_PA0_U0RX);
	GPIOPinConfigure(GPIO_PA1_U0TX);
	GPIOPinTypeUART(GPIO_PORTA_BASE, GPIO_PIN_0 | GPIO_PIN_1);

	UARTConfigSetExpClk(UART0_BASE, SysCtlClockGet(), 115200,
	(UART_CONFIG_WLEN_8 | UART_CONFIG_STOP_ONE | UART_CONFIG_PAR_NONE));
}

/****************************
 * System Control Functions *
 ****************************/
void twe_initSystem80MHz(void) {
	// Set system clock to 80 MHz (400MHz main PLL (divided by 5 - uses DIV400 bit)  [16MHz external xtal drives PLL]
	SysCtlClockSet(SYSCTL_SYSDIV_2_5 | SYSCTL_USE_PLL | SYSCTL_OSC_MAIN | SYSCTL_XTAL_16MHZ);
	ROM_SysCtlPeripheralClockGating(true); // Enable peripherals to operate when CPU is in sleep.
}

/******************
 * uDMA Functions *
 ******************/

/**
 * Initialize the uDMA controller at the system level.  It also enables it to continue
 * to run while the processor is in sleep.
 *
 * \note Individual uDMA channels need to be initialized seperately.
 *
 * \note Must also initialize the uDMA table immediately following this call.
 * i.e. ROM_uDMAControlBaseSet(uDMAcontrolTable);
 **/
void twe_initUDMAcontroller(void) {
	// Enable the uDMA controller at the system level.  Enable it to continue
	// to run while the processor is in sleep.
	ROM_SysCtlPeripheralEnable(SYSCTL_PERIPH_UDMA);
	ROM_SysCtlPeripheralSleepEnable(SYSCTL_PERIPH_UDMA);
	// Enable the uDMA controller error interrupt.  This interrupt will occur
	// if there is a bus error during a transfer.
	ROM_IntEnable(INT_UDMAERR);

    // Enable the uDMA controller.
    uDMAEnable();
    //IntEnable(INT_UDMA);
}


/**
 * uDMA Error Handler
 *
 * \note Need to add the ISR to the NVIC table in the row labeled "uDMA Error"
 *
 **/
void twe_uDMAErrorHandler(void) {
	uint32_t ui32Status;
	ui32Status = ROM_uDMAErrorStatusGet();
	if(ui32Status) {
		ROM_uDMAErrorStatusClear();
		g_DMAErrCount++;
	}
	uDMAChannelDisable(UDMA_CHANNEL_SSI0TX);
	SSIDMADisable(SSI0_BASE, SSI_DMA_RX | SSI_DMA_TX);
}

/******************
 * GPIO Functions *
 ******************/
void twe_initProcessingIndicator(void) {
	 SysCtlPeripheralEnable(TWE_PROCESSING_GPIO_PERIPH);
	 GPIOPinTypeGPIOOutput(TWE_PROCESSING_GPIO_BASE, TWE_PROCESSING_PIN);
	 GPIOPinWrite(TWE_PROCESSING_GPIO_BASE, TWE_PROCESSING_PIN, 0x00);
}

/***************************
 * RGB Extension Functions *
 ***************************/
#ifdef PART_TM4C123GH6PM // EK-TM4C123GXL

/**
 *
 * Initializes the Timer and GPIO functionality associated with the RGB LED
 *  to provide a solid LED output that doesn't blink.
 *
 * \note This function is an extension of the rgb driver of the EK-TM4C123GXL
 * Firmware Package provided by Texas Instruments.
 *
 * \note Doesn't use WTIMER5, which normally controls the blinking of the LEDs
 *
 * \note The RGB library does use Timer 0B and Timer 1A and Timer 1B
 *
 * \param ui32Enable enables RGB immediately if set.
 *
 * This function must be called during application initialization to
 * configure the GPIO pins to which the LEDs are attached.  It enables
 * the port used by the LEDs and configures each color's Timer. It optionally
 * enables the RGB LED by configuring the GPIO pins and starting the timers.
 *
 * \return None.
 *
 **/
void twe_RGBInitSolid(uint32_t ui32Enable) {
    // Enable the GPIO Port and Timer for each LED
    ROM_SysCtlPeripheralEnable(RED_GPIO_PERIPH);
    ROM_SysCtlPeripheralEnable(RED_TIMER_PERIPH);

    ROM_SysCtlPeripheralEnable(GREEN_GPIO_PERIPH);
    ROM_SysCtlPeripheralEnable(GREEN_TIMER_PERIPH);

    ROM_SysCtlPeripheralEnable(BLUE_GPIO_PERIPH);
    ROM_SysCtlPeripheralEnable(BLUE_TIMER_PERIPH);

    // Configure each timer for output mode
    HWREG(GREEN_TIMER_BASE + TIMER_O_CFG)   = 0x04;
    HWREG(GREEN_TIMER_BASE + TIMER_O_TAMR)  = 0x0A;
    HWREG(GREEN_TIMER_BASE + TIMER_O_TAILR) = 0xFFFF;

    HWREG(BLUE_TIMER_BASE + TIMER_O_CFG)   = 0x04;
    HWREG(BLUE_TIMER_BASE + TIMER_O_TBMR)  = 0x0A;
    HWREG(BLUE_TIMER_BASE + TIMER_O_TBILR) = 0xFFFF;

    HWREG(RED_TIMER_BASE + TIMER_O_CFG)   = 0x04;
    HWREG(RED_TIMER_BASE + TIMER_O_TBMR)  = 0x0A;
    HWREG(RED_TIMER_BASE + TIMER_O_TBILR) = 0xFFFF;

    // Invert the output signals.
    HWREG(RED_TIMER_BASE + TIMER_O_CTL)   |= 0x4000;
    HWREG(GREEN_TIMER_BASE + TIMER_O_CTL)   |= 0x40;
    HWREG(BLUE_TIMER_BASE + TIMER_O_CTL)   |= 0x4000;

    if(ui32Enable) {
        RGBEnable();
    }
}
/**
 *
 * Initializes the RGB and sets it to shine solid green.
 *
 * \note This function is an extension of the rgb driver of the EK-TM4C123GXL
 * firmware package provided by Texas Instruments.
 *
 * \note The RGB library does use Timer 0B and Timer 1A and Timer 1B
 *
 **/
void twe_RGBInitSetGreen(void) {
	uint32_t RGBcolor[3];
	twe_RGBInitSolid(0); //initialize the RGB for a solid output
	RGBIntensitySet(0.3f); // Set the intensity level (0.0f to 1.0f)
	RGBcolor[RED] =   0x0000;
	RGBcolor[GREEN] = 0xFFFF; // set the color to green
	RGBcolor[BLUE] =  0x0000;
	RGBColorSet(RGBcolor);

	RGBEnable();
}

#endif

/*****************
 * SSI Functions *
 *****************/

/**
  * Enables the TX end of transmission (EOT) feature that allows the TXFF interrupt
  *  to trigger immediately when a transmission completes
  *
  *  \note The EOT interrupt triggers the  TXFF interrupt
  *
  *  \note This enable is in the CR1 register rather than the interrupt mask (IM) register
  **/
void twe_SSIIntEnableEOT(uint32_t ui32Base) {

	// Check the argument.
	ASSERT(_SSIBaseValid(ui32Base));

	// Write to the control register
	HWREG(ui32Base + SSI_O_CR1) |= 0x00000010;
}

